<?php
if (!defined('DSACHECK_ACTIVE')) {
    die('Direct access not allowed');
}

/**
 * ارسال پاسخ JSON
 */
function response($success, $message, $data = null) {
    $response = [
        'success' => $success,
        'message' => $message
    ];
    
    if ($data !== null) {
        $response['data'] = $data;
    }

    // Temporary debug log: write response to error log to help diagnose client issues.
    if (defined('DSACHECK_ACTIVE') && DSACHECK_ACTIVE) {
        error_log('[dsacheck] RESPONSE: ' . json_encode($response));
    }

    echo json_encode($response);
    exit;
}

/**
 * بررسی محدودیت درخواست
 */
function check_rate_limit($ip) {
    $rate_limit_file = __DIR__ . '/../storage/rate_limits.json';
    $current_time = time();
    $time_window = 3600; // یک ساعت
    $max_requests = 100; // حداکثر تعداد درخواست در هر ساعت
    
    // خواندن فایل محدودیت‌ها
    $rate_limits = [];
    if (file_exists($rate_limit_file)) {
        $rate_limits = json_decode(file_get_contents($rate_limit_file), true);
    }
    
    // حذف رکوردهای منقضی شده
    foreach ($rate_limits as $key => $limit) {
        if ($limit['timestamp'] < ($current_time - $time_window)) {
            unset($rate_limits[$key]);
        }
    }
    
    // بررسی و به‌روزرسانی محدودیت IP فعلی
    if (!isset($rate_limits[$ip])) {
        $rate_limits[$ip] = [
            'count' => 1,
            'timestamp' => $current_time
        ];
    } else {
        $rate_limits[$ip]['count']++;
    }
    
    // ذخیره تغییرات
    file_put_contents($rate_limit_file, json_encode($rate_limits));
    
    return $rate_limits[$ip]['count'] <= $max_requests;
}

/**
 * ثبت لایسنس جدید
 */
function register_license($domain, $order_code, $plugin_key) {
    global $db;
    
    // اعتبارسنجی دامنه
    if (!filter_var('http://' . $domain, FILTER_VALIDATE_URL)) {
        return ['success' => false, 'message' => 'Invalid domain'];
    }
    
    // generate license key regardless
    $order = find_order($order_code);
    $license_key = generate_license_key($domain, $order_code);

    // Check if we already have a license row for this domain. If so, we'll update it instead of inserting duplicates.
    $stmtExisting = $db->prepare("SELECT * FROM licenses WHERE domain = ? LIMIT 1");
    $stmtExisting->execute([$domain]);
    $existingLicense = $stmtExisting->fetch(PDO::FETCH_ASSOC);

    try {
        // Create or update a license row for this registration attempt.
        // We do NOT store a 'status' column in licenses; permanence is controlled by orders.status.
        // If the related order exists and is 'disabled' (admin pre-added) or already 'approved', make this license permanent.
        if ($order) {
            // If order status is 'false' (rejected), do not allow updates from client side
            if ($order['status'] === 'false') {
                return ['success' => false, 'message' => 'Order rejected: cannot register license for this order'];
            }

            // Order exists. If it's 'disabled' (admin pre-added) or 'approved', convert to approved.
            if (in_array($order['status'], ['disabled','approved'], true)) {
                if ($existingLicense) {
                    // Update existing license row to mark approved
                    $stmt = $db->prepare("UPDATE licenses SET order_code = ?, license_key = ?, plugin_key = ?, approved_at = NOW(), pending_expires_at = NULL, rejected_at = NULL WHERE id = ?");
                    $ok = $stmt->execute([$order_code, $license_key, $plugin_key, $existingLicense['id']]);
                } else {
                    // Insert new approved license
                    $stmt = $db->prepare("INSERT INTO licenses (domain, order_code, license_key, plugin_key, approved_at, created_at) VALUES (?, ?, ?, ?, NOW(), NOW())");
                    $ok = $stmt->execute([$domain, $order_code, $license_key, $plugin_key]);
                }

                if (!empty($ok)) {
                    // Ensure orders.status is approved and mark used
                    $upd = $db->prepare("UPDATE orders SET status = 'approved', used = 1 WHERE code = ?");
                    $upd->execute([$order_code]);
                    return [
                        'success' => true,
                        'message' => 'License registered and approved (order existed)',
                        'data' => ['license_key' => $license_key, 'status' => 'approved', 'approved_at' => date('Y-m-d H:i:s'), 'permanent' => true]
                    ];
                }
                throw new PDOException('Failed to insert/update approved license');
            }

            // Order exists but is not approved (e.g., pending) -> create or update a temporary license for 24 hours
            if ($existingLicense) {
                $stmt = $db->prepare("UPDATE licenses SET order_code = ?, license_key = ?, plugin_key = ?, pending_expires_at = DATE_ADD(NOW(), INTERVAL 24 HOUR), approved_at = NULL, rejected_at = NULL WHERE id = ?");
                $ok = $stmt->execute([$order_code, $license_key, $plugin_key, $existingLicense['id']]);
                if ($ok) {
                    $stmt2 = $db->prepare("SELECT pending_expires_at FROM licenses WHERE id = ?");
                    $stmt2->execute([$existingLicense['id']]);
                    $row = $stmt2->fetch(PDO::FETCH_ASSOC);
                    return [
                        'success' => true,
                        'message' => 'License updated and pending admin approval (order exists but not approved)',
                        'data' => ['license_key' => $license_key, 'status' => 'pending', 'pending_expires_at' => $row['pending_expires_at'] ?? null]
                    ];
                }
                throw new PDOException('Failed to update pending license for existing order');
            } else {
                $stmt = $db->prepare("INSERT INTO licenses (domain, order_code, license_key, plugin_key, pending_expires_at, created_at) VALUES (?, ?, ?, ?, DATE_ADD(NOW(), INTERVAL 24 HOUR), NOW())");
                $ok = $stmt->execute([$domain, $order_code, $license_key, $plugin_key]);
                if ($ok) {
                    $stmt2 = $db->prepare("SELECT pending_expires_at FROM licenses WHERE license_key = ?");
                    $stmt2->execute([$license_key]);
                    $row = $stmt2->fetch(PDO::FETCH_ASSOC);
                    return [
                        'success' => true,
                        'message' => 'License created and pending admin approval (order exists but not approved)',
                        'data' => ['license_key' => $license_key, 'status' => 'pending', 'pending_expires_at' => $row['pending_expires_at'] ?? null]
                    ];
                }
                throw new PDOException('Failed to insert pending license for existing order');
            }
        }

        // Order not found -> create order with status 'pending' and create or update a temporary license valid for 24 hours
        $insOrder = $db->prepare("INSERT INTO orders (code, status, used, created_at) VALUES (?, 'pending', 0, NOW())");
        $insOrder->execute([$order_code]);

        if ($existingLicense) {
            $stmt = $db->prepare("UPDATE licenses SET order_code = ?, license_key = ?, plugin_key = ?, pending_expires_at = DATE_ADD(NOW(), INTERVAL 24 HOUR), approved_at = NULL, rejected_at = NULL WHERE id = ?");
            $ok = $stmt->execute([$order_code, $license_key, $plugin_key, $existingLicense['id']]);
            if ($ok) {
                $stmt2 = $db->prepare("SELECT pending_expires_at FROM licenses WHERE id = ?");
                $stmt2->execute([$existingLicense['id']]);
                $row = $stmt2->fetch(PDO::FETCH_ASSOC);
                return [
                    'success' => true,
                    'message' => 'License updated and pending admin approval (order created)',
                    'data' => ['license_key' => $license_key, 'status' => 'pending', 'pending_expires_at' => $row['pending_expires_at'] ?? null]
                ];
            }
            throw new PDOException('Failed to update pending license for unknown order');
        }

        $stmt = $db->prepare("INSERT INTO licenses (domain, order_code, license_key, plugin_key, pending_expires_at, created_at) VALUES (?, ?, ?, ?, DATE_ADD(NOW(), INTERVAL 24 HOUR), NOW())");
        $ok = $stmt->execute([$domain, $order_code, $license_key, $plugin_key]);
        if ($ok) {
            $stmt2 = $db->prepare("SELECT pending_expires_at FROM licenses WHERE license_key = ?");
            $stmt2->execute([$license_key]);
            $row = $stmt2->fetch(PDO::FETCH_ASSOC);
            return [
                'success' => true,
                'message' => 'License created and pending admin approval (order created)',
                'data' => ['license_key' => $license_key, 'status' => 'pending', 'pending_expires_at' => $row['pending_expires_at'] ?? null]
            ];
        }
        throw new PDOException('Failed to insert pending license and order for unknown order');
    } catch (PDOException $e) {
        error_log('[dsacheck] PDOException in register_license: ' . $e->getMessage());
        return ['success' => false, 'message' => 'Database error: ' . $e->getMessage()];
    }
}

/**
 * تایید اعتبار لایسنس
 */
function verify_license($domain, $license_key) {
    global $db;
    // بررسی دامنه و کلید لایسنس
    $stmt = $db->prepare("SELECT * FROM licenses WHERE domain = ? AND license_key = ?");
    $stmt->execute([$domain, $license_key]);
    $license = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$license) {
        return ['success' => false, 'message' => 'Invalid license'];
    }

    // If license links to an order, use orders.status as the source of truth
    if (!empty($license['order_code'])) {
        $order = find_order($license['order_code']);
        if ($order) {
            if ($order['status'] === 'approved') {
                // permanent
                return ['success' => true, 'message' => 'License is valid (approved)', 'data' => ['status' => 'approved', 'plugin_key' => $license['plugin_key'], 'approved_at' => $license['approved_at'] ?? null]];
            }
            if ($order['status'] === 'pending') {
                // check pending_expires_at on license
                if (!empty($license['pending_expires_at']) && strtotime($license['pending_expires_at']) > time()) {
                    return ['success' => true, 'message' => 'License pending approval (temporary active)', 'data' => ['status' => 'pending', 'pending_expires_at' => $license['pending_expires_at']]];
                }
                return ['success' => false, 'message' => 'Pending license expired', 'data' => ['status' => 'pending']];
            }
            if ($order['status'] === 'disabled') {
                // admin pre-added but not approved: treat as approved when user registers (this should be handled at registration time), but here deny if not approved
                return ['success' => false, 'message' => 'License not yet approved by admin', 'data' => ['status' => 'disabled']];
            }
            if ($order['status'] === 'false') {
                return ['success' => false, 'message' => 'License invalid', 'data' => ['status' => 'false']];
            }
        }
    }

    // Fallback: if license has approved_at (older flow), treat as approved
    if (!empty($license['approved_at'])) {
        return ['success' => true, 'message' => 'License is valid (approved)', 'data' => ['status' => 'approved', 'plugin_key' => $license['plugin_key'], 'approved_at' => $license['approved_at']]];
    }

    // If license has pending_expires_at and is still valid, treat as temporary
    if (!empty($license['pending_expires_at']) && strtotime($license['pending_expires_at']) > time()) {
        return ['success' => true, 'message' => 'License pending approval (temporary active)', 'data' => ['status' => 'pending', 'pending_expires_at' => $license['pending_expires_at']]];
    }

    return ['success' => false, 'message' => 'License not approved or expired'];
}

/**
 * بررسی کد سفارش
 */
function check_order_code($order_code, $domain = null) {
    global $db;
    
    // بررسی فرمت کد سفارش: قبول هرچیزی با طول 3 تا 12 کاراکتر (بدون فاصله)
    if (!preg_match('/^\S{3,12}$/', $order_code)) {
        return ['success' => false, 'message' => 'فرمت کد سفارش نامعتبر است (۳ تا ۱۲ کاراکتر)'];
    }

    // لاگ برای دیباگ
    error_log("Checking order code: " . $order_code . " for domain: " . $domain);

    // بررسی در دیتابیس
    $stmt = $db->prepare("SELECT * FROM orders WHERE code = ?");
    $stmt->execute([$order_code]);
    $order = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$order) {
        return ['success' => false, 'message' => 'Order not found', 'data' => null];
    }

    return ['success' => true, 'message' => 'Order found', 'data' => $order];
}

/**
 * Find an order row by code or return null
 */
function find_order($order_code) {
    global $db;
    $stmt = $db->prepare("SELECT * FROM orders WHERE code = ?");
    $stmt->execute([$order_code]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

/**
 * Find license by license_key
 */
function find_license_by_key($license_key) {
    global $db;
    $stmt = $db->prepare("SELECT * FROM licenses WHERE license_key = ?");
    $stmt->execute([$license_key]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

/**
 * Set license status (admin action)
 * $status should be one of: 'approved', 'rejected', 'pending'
 */
function set_license_status($license_key, $status, $admin_note = null) {
    global $db;
    // Allowed incoming statuses are: 'approved', 'rejected', 'pending'
    $allowed = array('approved','rejected','pending');
    if (!in_array($status, $allowed, true)) {
        return ['success' => false, 'message' => 'Invalid status'];
    }

    // Find the license and its order_code
    $lic = find_license_by_key($license_key);
    if (!$lic) {
        return ['success' => false, 'message' => 'License not found'];
    }

    // Update license timestamps and admin note as appropriate
    $fields = array();
    $params = array();
    if ($admin_note !== null) {
        $fields[] = 'admin_note = ?';
        $params[] = $admin_note;
    }

    if ($status === 'approved') {
        $fields[] = 'approved_at = NOW()';
    } elseif ($status === 'rejected') {
        $fields[] = 'rejected_at = NOW()';
    } elseif ($status === 'pending') {
        $fields[] = "pending_expires_at = DATE_ADD(NOW(), INTERVAL 24 HOUR)";
    }

    // Build and execute license update if we have fields
    if (!empty($fields)) {
        $sql = "UPDATE licenses SET " . implode(', ', $fields) . " WHERE license_key = ?";
        $params[] = $license_key;
        $stmt = $db->prepare($sql);
        $stmt->execute($params);
    }

    // Sync status to orders table if license links to an order
    if (!empty($lic['order_code'])) {
        if ($status === 'approved') {
            $upd = $db->prepare("UPDATE orders SET status = 'approved', used = 1 WHERE code = ?");
            $upd->execute([$lic['order_code']]);
            // Also ensure license has approved_at if not set
            $db->prepare("UPDATE licenses SET approved_at = NOW() WHERE license_key = ? AND approved_at IS NULL")->execute([$license_key]);
        } elseif ($status === 'rejected') {
            $upd = $db->prepare("UPDATE orders SET status = 'false' WHERE code = ?");
            $upd->execute([$lic['order_code']]);
            $db->prepare("UPDATE licenses SET rejected_at = NOW() WHERE license_key = ? AND rejected_at IS NULL")->execute([$license_key]);
        } elseif ($status === 'pending') {
            $upd = $db->prepare("UPDATE orders SET status = 'pending' WHERE code = ?");
            $upd->execute([$lic['order_code']]);
            $db->prepare("UPDATE licenses SET pending_expires_at = DATE_ADD(NOW(), INTERVAL 24 HOUR) WHERE license_key = ?")->execute([$license_key]);
        }
    }

    return ['success' => true, 'message' => 'Status updated and orders synced'];
}

/**
 * تولید کلید لایسنس
 */
function generate_license_key($domain, $order_code) {
    $data = $domain . $order_code . time() . uniqid();
    return hash('sha256', $data);
}