<?php
define('DSACHECK_ACTIVE', true);
require_once 'config.php';
require_once 'includes/functions.php';
require_once 'includes/database.php';

// Enable error logging
error_reporting(E_ALL);
// In production keep errors logged but do not display them to users
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/error.log');

// Read raw input once for later parsing
$input_data = file_get_contents('php://input');

// تنظیم هدرهای CORS و امنیتی
header('Access-Control-Allow-Origin: *');
header('Content-Type: application/json; charset=UTF-8');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Max-Age: 3600');
header('Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With');

// بررسی متد درخواست
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    response(false, 'Invalid request method');
}

// دریافت داده‌های خام: ابتدا سعی می‌کنیم JSON را بخوانیم، در غیر اینصورت از فرم استفاده می‌کنیم
// Note: php://input can only be read once, we already read it into $input_data above, reuse it.
$raw = $input_data;
$input = json_decode($raw, true);
if (!$input) {
    // fallback to parsed POST (form-encoded)
    $input = $_POST;
}

// Temporary debugging: log raw input and parsed input to server error log to diagnose client issues.
error_log('[dsacheck] Remote IP: ' . ($_SERVER['REMOTE_ADDR'] ?? 'unknown'));
error_log('[dsacheck] Raw input: ' . substr($raw ?? '', 0, 2000));
// log parsed input structure (safe truncation)
error_log('[dsacheck] Parsed input: ' . print_r($input, true));
// log headers if available
if (function_exists('getallheaders')) {
    $h = getallheaders();
    error_log('[dsacheck] Headers: ' . print_r($h, true));
}

// Minimal request logging: only log when PHP error logging is enabled; avoid dumping request bodies in production.
// If deeper diagnostics are needed, re-enable debug_received.txt temporarily.

// بررسی داده‌های ورودی
if (!$input || !isset($input['action'])) {
    response(false, 'Invalid request data');
}

// بررسی محدودیت درخواست
if (!check_rate_limit($_SERVER['REMOTE_ADDR'])) {
    response(false, 'Rate limit exceeded');
}

// پردازش درخواست بر اساس نوع عملیات
switch ($input['action']) {
    case 'test_connection':
        // تست اتصال به سرور
        response(true, 'اتصال به سرور موفقیت‌آمیز بود', array(
            'server_time' => date('Y-m-d H:i:s'),
            'received_data' => $input
        ));
        break;

    case 'db_test':
        // تست اتصال به دیتابیس
        try {
            $ok = $db->query('SELECT 1')->fetch();
            response(true, 'Database connection OK', array('db_result' => $ok));
        } catch (Exception $e) {
            response(false, 'Database error: ' . $e->getMessage());
        }
        break;

    case 'verify':
        // تایید لایسنس: می‌توانیم توسط license_key بررسی کنیم (preferred) یا در صورت نبود از order_code استفاده کنیم
        $domain = $input['domain'] ?? null;
        if (!empty($input['license_key'])) {
            if (empty($domain)) {
                response(false, 'دامنه ارسال نشده است');
            }
            $result = verify_license($domain, $input['license_key']);
            response($result['success'], $result['message'], $result['data'] ?? null);
        }

        // fallback: verify by order_code (note: this checks orders table)
        if (!isset($input['order_code'], $input['domain'])) {
            response(false, 'کد سفارش یا دامنه ارسال نشده است');
        }

        $result = check_order_code($input['order_code'], $input['domain']);
        response($result['success'], $result['message'], $result['data'] ?? null);
        break;

    case 'register':
        // ثبت لایسنس از سفارش (اختیاری)
        // Accept either 'order_code' (preferred) or legacy 'license_key' sent from older plugin versions
        $order_code = $input['order_code'] ?? ($input['license_key'] ?? null);
        $domain = $input['domain'] ?? null;
        $plugin_key = $input['plugin_key'] ?? null;

        if (empty($order_code) || empty($domain) || empty($plugin_key)) {
            response(false, 'Missing required fields: order_code/domain/plugin_key');
        }

        $result = register_license($domain, $order_code, $plugin_key);
        response($result['success'], $result['message'], $result['data'] ?? null);
        break;

    case 'check_order':
        // بررسی کد سفارش
        if (!isset($input['order_code'])) {
            response(false, 'Missing order code');
        }
        
        $result = check_order_code($input['order_code']);
        response($result['success'], $result['message'], $result['data'] ?? null);
        break;

    case 'admin_update_license_status':
        // Protected admin action to update license status (approve/reject/pending)
        $secret = $input['secret'] ?? null;
        $license_key = $input['license_key'] ?? null;
        $status = $input['status'] ?? null;
        $note = $input['admin_note'] ?? null;

        if (empty($secret) || $secret !== API_SECRET_KEY) {
            response(false, 'Unauthorized');
        }
        if (empty($license_key) || empty($status)) {
            response(false, 'Missing license_key or status');
        }

        $result = set_license_status($license_key, $status, $note);
        response($result['success'], $result['message'], $result['data'] ?? null);
        break;

    default:
        response(false, 'Invalid action');
}